/* kXML
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License
 * on the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of kXML is Stefan Haustein. Copyright (C)
 * 2000, 2001 Stefan Haustein, D-46045 Oberhausen (Rhld.),
 * Germany. All Rights Reserved.
 *
 * Contributor(s): Paul Palaszewski, Wilhelm Fitzpatrick, 
 *                 Eric Foster-Johnson
 *
 * */

package org.kxml.io;

import java.io.*;
import java.util.*;
import org.kxml.*;

public abstract class AbstractXmlReader {

    protected int type;
    protected String name;
    protected String namespace;
    protected String text;

    protected Attribute [] attributes = new Attribute [8];
    protected int attributeCount;
    protected boolean degenerated;

    protected State current = 
	new State (null, 
		   PrefixMap.DEFAULT, //null, null, 
		   null);


    public Hashtable characterEntityTable;
    public boolean processNamespaces = true;
    public boolean relaxed;

    
    protected void adjustNamespace () throws IOException {

	boolean any = false;

	for (int i = attributeCount-1; i >= 0; i--) {
	    Attribute attr = attributes [i];
	    String attrName = attr.getName ();
	    int cut = attrName.indexOf (':');
	    String prefix;
	    
	    if (cut != -1) {
		prefix = attrName.substring (0, cut);
		attrName = attrName.substring (cut+1);
	    }
	    else if (attrName.equals ("xmlns")) {
		prefix = attrName;
		attrName = "";
	    } 
	    else continue;
	    
	    if (!prefix.equals ("xmlns")) {
		if (!prefix.equals ("xml")) any = true;
	    }
	    else {
		current.prefixMap = new PrefixMap 
		    (current.prefixMap, attrName, attr.getValue ());
		
		attributeCount--;
		for (int j = i; j < attributeCount; j++)
		    attributes [j] = attributes [j+1];
	    }
	}	    
	
	
	if (any) {
	    for (int i = 0; i < attributeCount; i++) {
		Attribute attr = attributes [i];
		String attrName = attr.getName ();
		int cut = attrName.indexOf (':');
		
		if (cut == 0) 
		    exception
			("illegal attribute name: "+attrName);
		
		else if (cut != -1) {		    
		    String attrPrefix = attrName.substring (0, cut);
		    if (!attrPrefix.equals ("xml")) {
			attrName = attrName.substring (cut+1);
			
			String attrNs = current.prefixMap.getNamespace 
			    (attrPrefix);
			
			if (attrNs == null) 
			    exception
				("Undefined Prefix: "+attrPrefix + " in ");
			
			attributes [i] =
			    new Attribute 
				(attrNs, 
				 attrName, attr.getValue ());
		    }
		}
	    }
	}
    	    
	int cut = name.indexOf (':');
	    
	String prefix = "";

	if (cut == 0)
	    exception
		("illegal tag name: "+ name);
	else if (cut != -1) {
	    prefix = name.substring (0, cut);
	    this.name = name.substring (cut+1);
	}
	
	this.namespace = current.prefixMap.getNamespace (prefix);
	
	if (this.namespace == null) {
	    if (prefix.length () != 0) 
		exception
		    ("undefined prefix: "+prefix+" in "
		     +current.prefixMap);
	    this.namespace = Xml.NO_NAMESPACE;
	}
    }

    /** returns the event type integer constant assigned to the
        current XML token.  Possible type values are Xml.START_TAG,
        Xml.END_TAG, Xml.TEXT, Xml.PROCESSING_INSTRUCTION,
        Xml.COMMENT, Xml.DOCTYPE, and Xml.END_DOCUMENT */

    public int getType () {
	return type;
    }


    public boolean getDegenerated () {
	return degenerated;
    }


    /** returns the (local) tag name of the element if the token type
        is Xml.START_TAG or XML.END_TAG.  For all other types,  null is
	returned*/

    public String getName () {
	return name;
    }


    /** returns the namespace of the element if the token type
        is Xml.START_TAG or XML.END_TAG. For all other types,  null is
	returned. */

    public String getNamespace () {
	return namespace;
    }


    /** If the event type is TEXT, PROCESSING_INSTRUCTION,
	or DOCTYPE, the corresponding string is returned. For
	all othe event types, null is returned. */

    public String getText () {
	return text;
    }


    
    public PrefixMap getPrefixMap () {
	return current.prefixMap;
    }
    


    public int getAttributeCount () {
	return attributeCount;
    }

    /** In the event type is START_TAG, this method returns the
        attribute at the given index position. For all other event
        types, or if the index is out of range, an
        IndexOutOfBoundsException will be thrown. */

    public Attribute getAttribute (int i) {
	if (i < 0 || i >= attributeCount) 
	    throw new IndexOutOfBoundsException ();

	return attributes [i];
    }


  /** returns the local attribute with the given name.  convenience
	method for getAttribute (Xml.NO_NAMESPACE, name); */

    public Attribute getAttribute (String name) {
	return getAttribute (Xml.NO_NAMESPACE, name);
    }


   /** returns the local attribute with the given qualified name.
        Please use null as placeholder for any namespace or
        Xml.NO_NAMESPACE for no namespace. */

    public Attribute getAttribute (String namespace, String name) {
	for (int i = 0; i < attributeCount; i++) {
	    	    
	    if (attributes [i].getName ().equals (name) 
		&& (namespace == null 
		    || namespace.equals (attributes [i].getNamespace ())))
		
		return attributes [i];  
	}
	return null;
    }


    /** Returns the value of the attribute with the given name.
	Throws an exception if not instanceof StartTag or if not
	existing. In order to get a null value for not existing
	attributes, please call getValueDefault (attrName, null)
	instead. */

    public String getValue (String attrName) {
	Attribute attr = getAttribute (Xml.NO_NAMESPACE, attrName);
	if (attr == null) throw new RuntimeException 
	    ("Attribute "+ attrName + " in " + this + " expected!");
	return attr.getValue ();
    }


    /** Returns the given attribute value, or the given default value
	if the attribute is not existing. */

    public String getValueDefault (String attrName, String deflt) {
	Attribute attr = getAttribute (Xml.NO_NAMESPACE, attrName);
	return attr == null ? deflt : attr.getValue ();
    }

 

    /** advances to the next xml token. Returns the token type code
	@return the token type code. */

    public int next () throws IOException {
	nextImpl ();
	if (type == Xml.START_TAG) {

	    current = new State (current, current.prefixMap, name);

	    if (namespace == null) { 
		if (processNamespaces) 
		    adjustNamespace ();
		else
		    namespace = Xml.NO_NAMESPACE;
	    }
	    
	    current.name = name;
	    current.namespace = namespace;
	}
	else if (type == Xml.END_TAG) {

	    while (true) {
		if (current.prev == null) 
		    exception
			("tagstack empty parsing </"+name+">");
		
		if (current.tag.equals (name)) break; 
		if (!relaxed) exception
		    ("StartTag <"+current.tag
		     +"> does not match end tag </" + name + ">");
		if (current.tag.toLowerCase ().equals (name.toLowerCase ())) 
		    break;

		current = current.prev;
	    }

	    name = current.name;
	    namespace = current.namespace;
	    current = current.prev;
	}

	return type;
    }
    
    /** works like next, but skips over 
	comments, processing instructions, whitespace and the doctype */

    public int skip () throws IOException {
	while (true) {
	    switch (next ()) {
	    case Xml.END_TAG:
	    case Xml.END_DOCUMENT:
	    case Xml.TEXT:
	    case Xml.START_TAG:
		return type;
	    }
	}   
    }


    public void require (int type, 
			 String namespace, String name) throws IOException {

	if (this.type != type
	    || (namespace != null 
		&& !namespace.equals (this.namespace))
	    || (name != null 
		&& !name.equals (this.name)))
	    exception ("unexpected token!");
    }
	

    public String readText () throws IOException {
	StringBuffer buf = new StringBuffer ();
	while (true) {
	    switch (type) { 
	    case Xml.END_DOCUMENT:
	    case Xml.END_TAG:
	    case Xml.START_TAG:
		return buf.toString();

	    case Xml.TEXT:
	    case Xml.WHITESPACE:
		buf.append (getText ());
	    }
	    next ();
	}

    }


    protected abstract void nextImpl () throws IOException;

    protected void exception (String name) throws IOException {
	throw new ParseException 
	    (name, null, getLineNumber (), getColumnNumber ());
    }


    public int getLineNumber () {
	return -1;
    }
    
    public int getColumnNumber () {
	return -1;
    }
}
